<?php

namespace App\Http\Controllers\Karyawan;

use App\Http\Controllers\Controller;
use App\Models\Karyawan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class SettingsController extends Controller
{
    /**
     * Display settings page
     */
    public function index()
    {
        $karyawan = Karyawan::with(['departemen', 'jabatan'])
            ->where('user_id', Auth::id())
            ->firstOrFail();

        return view('karyawan.settings.index', compact('karyawan'));
    }

    /**
     * Update profile photo
     */
    public function updatePhoto(Request $request)
    {
        $request->validate([
            'foto_profil' => 'required|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        try {
            $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

            // Delete old photo if exists
            if ($karyawan->foto_profil) {
                Storage::disk('public')->delete($karyawan->foto_profil);
            }

            // Store new photo
            $file = $request->file('foto_profil');
            $fileName = 'profile_' . $karyawan->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('profiles', $fileName, 'public');

            // Update database
            $karyawan->update([
                'foto_profil' => $path
            ]);

            return redirect()->route('karyawan.settings.index')
                ->with('success', ' Foto profil berhasil diperbarui!');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', ' Gagal mengupdate foto profil: ' . $e->getMessage());
        }
    }

    /**
     * Update email
     */
    public function updateEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|email|unique:users,email,' . Auth::id(),
            'password' => 'required'
        ]);

        try {
            $user = Auth::user();

            // Verify current password
            if (!Hash::check($request->password, $user->password)) {
                return redirect()->back()
                    ->with('error', ' Password yang Anda masukkan salah!');
            }

            // Update email
            $user->update([
                'email' => $request->email
            ]);

            return redirect()->route('karyawan.settings')
                ->with('success', ' Email berhasil diperbarui!');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', ' Gagal mengupdate email: ' . $e->getMessage());
        }
    }

    /**
     * Update password
     */
    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => [
                'required',
                'confirmed',
                Password::min(8)
                    ->mixedCase()
                    ->numbers()
                    ->symbols()
            ],
        ], [
            'new_password.min' => 'Password minimal 8 karakter',
            'new_password.mixed_case' => 'Password harus mengandung huruf besar dan kecil',
            'new_password.numbers' => 'Password harus mengandung angka',
            'new_password.symbols' => 'Password harus mengandung karakter khusus',
            'new_password.confirmed' => 'Konfirmasi password tidak cocok',
        ]);

        try {
            $user = Auth::user();

            // Verify current password
            if (!Hash::check($request->current_password, $user->password)) {
                return redirect()->back()
                    ->with('error', ' Password saat ini salah!');
            }

            // Check if new password is same as current
            if (Hash::check($request->new_password, $user->password)) {
                return redirect()->back()
                    ->with('error', ' Password baru tidak boleh sama dengan password lama!');
            }

            // Update password
            $user->update([
                'password' => Hash::make($request->new_password)
            ]);

            // Optional: Logout user and redirect to login
            // Auth::logout();
            // return redirect()->route('login')->with('success', 'Password berhasil diubah. Silakan login kembali.');

            return redirect()->route('karyawan.settings')
                ->with('success', ' Password berhasil diperbarui!');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', ' Gagal mengupdate password: ' . $e->getMessage());
        }
    }
}
